# IBrandBiz – Outline & Sections (Core Build)

Here’s the expanded spec + scaffold with more structured templates and section support.

---

## Default Plan Outline (Must-Have Sections)
1. **Executive Summary**
2. **Company Overview**
   - Mission & Vision
   - Legal Structure
   - Founding Team
3. **Products & Services**
   - Product/Service Description
   - Value Proposition
   - Roadmap
4. **Market Analysis**
   - Industry Overview
   - Target Market (TAM/SAM/SOM)
   - Competitive Landscape
   - SWOT Analysis (Template)
5. **Marketing & Sales**
   - Customer Personas (Template)
   - Go-to-Market Strategy
   - Pricing Model (Template)
   - Sales Channels
6. **Operations Plan**
   - Key Activities
   - Resources & Suppliers
   - Technology Stack
7. **Organization & Management**
   - Org Chart
   - Roles & Responsibilities
   - Advisors
8. **Financial Plan**
   - Revenue Model
   - Cost Structure
   - 3-Year Projections (tables)
   - Break-even Analysis
   - Funding Requirements

---

## Templates (Fill-in-the-Blank)

### SWOT Analysis
- **Strengths:** [List internal advantages]
- **Weaknesses:** [List internal challenges]
- **Opportunities:** [List external trends/opportunities]
- **Threats:** [List external risks]

### Customer Persona
- **Name:** [Persona label]
- **Demographics:** Age, gender, location
- **Goals:** [Primary objectives]
- **Frustrations:** [Pain points]
- **Buying Triggers:** [Motivators]

### TAM / SAM / SOM
- **TAM (Total Addressable Market):** [Global demand]
- **SAM (Serviceable Available Market):** [Reachable market]
- **SOM (Serviceable Obtainable Market):** [Market share goal]

### Pricing Table
| Tier | Features | Price |
|------|----------|-------|
| Free | Basic access | $0 |
| Pro | Premium features | $49/mo |
| Enterprise | Custom features & support | Contact us |

### Porter’s Five Forces
1. Threat of New Entrants
2. Bargaining Power of Suppliers
3. Bargaining Power of Buyers
4. Threat of Substitutes
5. Industry Rivalry

---

## Drag-and-Drop Editor Features
- Reorder sections and sub-sections (tree-based)
- Add new section or template block
- Duplicate/delete sections
- Inline rename

---

## AI Prompt Hooks
- **Outline Generator:**
  > “Generate a structured business plan outline for [industry], with 6–10 sections and optional subsections.”

- **Section Drafting:**
  > “Write a [tone: Professional/Friendly/Bold/Minimal] draft for the section [name], using inputs: [user notes]. Return markdown paragraphs only.”

---

## Next Dev Steps
- Persist outline (localStorage → API)
- UI rendering with shadcn (Cards, Tree View)
- Live preview of templates → markdown → rendered prose
- Wire export (PDF/DOCX/Google Docs)

---

/* =========================
   V2 ADDITIONS – nested subsections, persistence, more templates
   ========================= */

/* =========================
   File: src/state/usePlanStore.v2.ts
   ========================= */
import { create as createZ } from "zustand";
import { nanoid as nid } from "nanoid";
import type { PlanState, PlanSection, SectionKind, SectionId } from "../types/plan";

const V2_STORAGE_KEY = "ibrandbiz.plan.v2";

const V2_DEFAULT_SECTIONS: PlanSection[] = [
  { id: nid(), kind: "executive-summary", title: "Executive Summary", content: "" },
  { id: nid(), kind: "company-overview", title: "Company Overview", content: "" },
  { id: nid(), kind: "products-services", title: "Products & Services", content: "" },
  { id: nid(), kind: "market-analysis", title: "Market Analysis", content: "" },
  { id: nid(), kind: "marketing-sales", title: "Marketing & Sales", content: "" },
  { id: nid(), kind: "operations", title: "Operations Plan", content: "" },
  { id: nid(), kind: "org-management", title: "Organization & Management", content: "" },
  { id: nid(), kind: "financials", title: "Financial Plan", content: "" },
];

function v2Load(): PlanState {
  try {
    const raw = localStorage.getItem(V2_STORAGE_KEY);
    if (raw) return JSON.parse(raw) as PlanState;
  } catch {}
  return { planId: nid(), title: "Untitled Business Plan", sections: V2_DEFAULT_SECTIONS };
}

type V2Store = PlanState & {
  addSection(kind?: SectionKind, title?: string): SectionId;
  addChildSection(parentId: SectionId, title?: string): SectionId | null;
  removeSection(id: SectionId): void;
  removeChildSection(parentId: SectionId, childId: SectionId): void;
  reorderSections(idsInOrder: SectionId[]): void;
  reorderChildSections(parentId: SectionId, idsInOrder: SectionId[]): void;
  updateSection(id: SectionId, patch: Partial<PlanSection>): void;
  updateChildSection(parentId: SectionId, childId: SectionId, patch: Partial<PlanSection>): void;
  duplicateSection(id: SectionId): SectionId | null;
  resetToDefault(): void;
};

export const usePlanStoreV2 = createZ<V2Store>((set, get) => ({
  ...v2Load(),
  addSection: (kind = "custom", title = "New Section") => {
    const sec: PlanSection = { id: nid(), kind, title, content: "", children: [] };
    set((s) => ({ sections: [...s.sections, sec] }));
    v2Persist();
    return sec.id;
  },
  addChildSection: (parentId, title = "New Subsection") => {
    set((s) => ({
      sections: s.sections.map((sec) =>
        sec.id === parentId
          ? { ...sec, children: [...(sec.children || []), { id: nid(), kind: "custom", title, content: "" }] }
          : sec
      ),
    }));
    v2Persist();
    const p = get().sections.find((x) => x.id === parentId);
    const last = p?.children?.[p.children.length - 1];
    return last?.id || null;
  },
  removeSection: (id) => { set((s) => ({ sections: s.sections.filter((x) => x.id !== id) })); v2Persist(); },
  removeChildSection: (parentId, childId) => {
    set((s) => ({
      sections: s.sections.map((sec) => sec.id === parentId ? { ...sec, children: (sec.children||[]).filter((c)=>c.id!==childId)}: sec)
    }));
    v2Persist();
  },
  reorderSections: (ids) => { set((s) => ({ sections: ids.map((id) => s.sections.find((x)=>x.id===id)!).filter(Boolean) })); v2Persist(); },
  reorderChildSections: (parentId, ids) => {
    set((s) => ({
      sections: s.sections.map((sec) => {
        if (sec.id !== parentId) return sec;
        const current = sec.children || [];
        const out = ids.map((id) => current.find((c)=>c.id===id)!).filter(Boolean);
        return { ...sec, children: out };
      })
    }));
    v2Persist();
  },
  updateSection: (id, patch) => { set((s)=>({ sections: s.sections.map((sec)=>sec.id===id?{...sec, ...patch}:sec) })); v2Persist(); },
  updateChildSection: (parentId, childId, patch) => {
    set((s)=>({ sections: s.sections.map((sec)=> sec.id!==parentId? sec : { ...sec, children: (sec.children||[]).map((c)=> c.id===childId?{...c, ...patch}:c) }) }));
    v2Persist();
  },
  duplicateSection: (id) => {
    const src = get().sections.find((x)=>x.id===id); if (!src) return null;
    const dupe: PlanSection = { ...src, id: nid(), title: `${src.title} (Copy)` };
    set((s)=>({ sections: [...s.sections, dupe] })); v2Persist(); return dupe.id;
  },
  resetToDefault: () => {
    const state = { planId: nid(), title: "Untitled Business Plan", sections: V2_DEFAULT_SECTIONS };
    set(state); try { localStorage.setItem(V2_STORAGE_KEY, JSON.stringify(state)); } catch {}
  },
}));

function v2Persist(){
  try {
    const { planId, title, sections } = usePlanStoreV2.getState();
    localStorage.setItem(V2_STORAGE_KEY, JSON.stringify({ planId, title, sections }));
  } catch {}
}

/* =========================
   File: src/templates/structured.v2.ts
   ========================= */
import type { StructuredTemplate } from "../types/plan";

export const V2_TEMPLATES: StructuredTemplate[] = [
  {
    key: "porter5",
    name: "Porter’s Five Forces",
    description: "Analyze industry competitiveness.",
    fields: [
      { id: "threat_new", label: "Threat of New Entrants", type: "textarea" },
      { id: "bargain_sup", label: "Bargaining Power of Suppliers", type: "textarea" },
      { id: "bargain_buy", label: "Bargaining Power of Buyers", type: "textarea" },
      { id: "threat_sub", label: "Threat of Substitutes", type: "textarea" },
      { id: "rivalry", label: "Industry Rivalry", type: "textarea" },
    ],
  },
  {
    key: "tam_sam_som",
    name: "TAM / SAM / SOM",
    description: "Market sizing overview.",
    fields: [
      { id: "tam", label: "TAM (Total Addressable Market)", type: "textarea", placeholder: "Size, assumptions, sources" },
      { id: "sam", label: "SAM (Serviceable Available Market)", type: "textarea" },
      { id: "som", label: "SOM (Serviceable Obtainable Market)", type: "textarea" },
      { id: "sources", label: "Sources / Citations", type: "textarea" },
    ],
  },
  {
    key: "pricing_table",
    name: "Pricing Table",
    description: "List tiers or SKUs with price & notes.",
    fields: [
      { id: "rows", label: "Rows (Name – Price – Notes)", type: "textarea", placeholder: "Basic – $19/mo – For starters
Pro – $49/mo – Most popular
Enterprise – $199/mo – Custom SLAs" },
    ],
  },
];

export function renderV2TemplateMarkdown(key: string, data: Record<string,string>): string {
  if (key === "porter5") return `### Porter’s Five Forces

**Threat of New Entrants**
${data.threat_new||"-"}

**Bargaining Power of Suppliers**
${data.bargain_sup||"-"}

**Bargaining Power of Buyers**
${data.bargain_buy||"-"}

**Threat of Substitutes**
${data.threat_sub||"-"}

**Industry Rivalry**
${data.rivalry||"-"}`;
  if (key === "tam_sam_som") return `### Market Size: TAM / SAM / SOM

**TAM**
${data.tam||"-"}

**SAM**
${data.sam||"-"}

**SOM**
${data.som||"-"}

**Sources**
${data.sources||"-"}`;
  if (key === "pricing_table") {
    const rows = (data.rows||"").split(/
+/).filter(Boolean);
    const table = ["| Name | Price | Notes |", "|---|---:|---|"]
      .concat(rows.map((r)=>{ const [name="", price="", notes=""] = r.split(" – "); return `| ${name.trim()} | ${price.trim()} | ${notes.trim()} |`; }))
      .join("
");
    return `### Pricing

${table}`;
  }
  return "";
}

/* =========================
   File: src/components/OutlineEditor.v2.tsx
   ========================= */
import React from "react";
import { DndContext, closestCenter, PointerSensor, useSensor, useSensors } from "@dnd-kit/core";
import { arrayMove, SortableContext, verticalListSortingStrategy } from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { useSortable } from "@dnd-kit/sortable";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

function Row({ id, title, parentId }: { id: string; title: string; parentId?: string }) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({ id });
  const update = usePlanStoreV2((s)=> s.updateSection);
  const updateChild = usePlanStoreV2((s)=> s.updateChildSection);
  const remove = usePlanStoreV2((s)=> s.removeSection);
  const removeChild = usePlanStoreV2((s)=> s.removeChildSection);
  const addChild = usePlanStoreV2((s)=> s.addChildSection);

  const style: React.CSSProperties = { transform: CSS.Transform.toString(transform), transition, opacity: isDragging?0.6:1 };
  const onRename = (val:string)=> parentId ? updateChild(parentId, id, { title: val }) : update(id, { title: val });
  const onDelete = ()=> parentId ? removeChild(parentId, id) : remove(id);

  return (
    <div ref={setNodeRef} style={style} className="flex items-center justify-between rounded-xl border p-2">
      <div className="flex items-center gap-3">
        <button {...attributes} {...listeners} className="cursor-grab">≡</button>
        <input className="border-none outline-none bg-transparent text-base" defaultValue={title} onBlur={(e)=> onRename(e.currentTarget.value)} />
      </div>
      <div className="flex gap-2">
        {!parentId && (<button onClick={()=> addChild(id)} className="px-2 py-1 rounded-lg border">Add Subsection</button>)}
        <button onClick={onDelete} className="px-2 py-1 rounded-lg border text-red-600">Delete</button>
      </div>
    </div>
  );
}

export function OutlineEditorV2(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const reorder = usePlanStoreV2((s)=> s.reorderSections);
  const reorderChild = usePlanStoreV2((s)=> s.reorderChildSections);
  const sensors = useSensors(useSensor(PointerSensor));

  function onDragEndTop(event:any){
    const { active, over } = event; if (!over || active.id===over.id) return;
    const ids = sections.map((s)=> s.id); const oldIndex = ids.indexOf(active.id); const newIndex = ids.indexOf(over.id);
    reorder(arrayMove(ids, oldIndex, newIndex));
  }
  return (
    <div className="space-y-3">
      <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEndTop}>
        <SortableContext items={sections.map((s)=> s.id)} strategy={verticalListSortingStrategy}>
          {sections.map((s)=> (
            <div key={s.id} className="rounded-2xl border p-3">
              <Row id={s.id} title={s.title} />

              {(s.children||[]).length>0 && (
                <ChildList parentId={s.id} />
              )}
            </div>
          ))}
        </SortableContext>
      </DndContext>
    </div>
  );

  function ChildList({ parentId }:{ parentId: string }){
    const parent = sections.find((x)=> x.id===parentId);
    const children = parent?.children || [];

    function onDragEndChild(event:any){
      const { active, over } = event; if (!over || active.id===over.id) return;
      const ids = children.map((c)=> c.id); const oldIndex = ids.indexOf(active.id); const newIndex = ids.indexOf(over.id);
      reorderChild(parentId, arrayMove(ids, oldIndex, newIndex));
    }
    return (
      <div className="mt-3 ml-7">
        <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEndChild}>
          <SortableContext items={children.map((c)=> c.id)} strategy={verticalListSortingStrategy}>
            <div className="space-y-2">
              {children.map((c)=> (<Row key={c.id} id={c.id} title={c.title} parentId={parentId} />))}
            </div>
          </SortableContext>
        </DndContext>
      </div>
    );
  }
}

/* =========================
   File: src/components/StructuredForm.v2.tsx
   ========================= */
import React, { useState } from "react";
import { V2_TEMPLATES, renderV2TemplateMarkdown } from "../templates/structured.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

export function StructuredFormV2({ targetSectionId }:{ targetSectionId: string }){
  const [selectedKey, setSelectedKey] = useState<string>(V2_TEMPLATES[0].key);
  const [data, setData] = useState<Record<string,string>>({});
  const update = usePlanStoreV2((s)=> s.updateSection);

  const tpl = V2_TEMPLATES.find((t)=> t.key===selectedKey)!;
  function insert(){
    const md = renderV2TemplateMarkdown(tpl.key, data);
    update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
  }

  return (
    <div className="space-y-3">
      <select value={selectedKey} onChange={(e)=> setSelectedKey(e.target.value)} className="border rounded-xl p-2">
        {V2_TEMPLATES.map((t)=> (<option key={t.key} value={t.key}>{t.name}</option>))}
      </select>
      <div className="grid gap-2">
        {tpl.fields.map((f)=> (
          <label key={f.id} className="grid gap-1">
            <span className="text-sm text-gray-600">{f.label}</span>
            {f.type === "textarea" ? (
              <textarea rows={3} className="rounded-xl border p-2" placeholder={f.placeholder} onChange={(e)=> setData((d)=> ({...d, [f.id]: e.target.value}))} />
            ) : (
              <input className="rounded-xl border p-2" placeholder={f.placeholder} onChange={(e)=> setData((d)=> ({...d, [f.id]: e.target.value}))} />
            )}
          </label>
        ))}
      </div>
      <button onClick={insert} className="rounded-2xl border px-3 py-2">Insert into Section</button>
    </div>
  );
}

/* =========================
   File: src/App.v2.tsx (integration sketch)
   ========================= */
import React from "react";
import { OutlineEditorV2 } from "./components/OutlineEditor.v2";
import { usePlanStoreV2 } from "./state/usePlanStore.v2";
import { StructuredFormV2 } from "./components/StructuredForm.v2";

export default function AppV2(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const update = usePlanStoreV2((s)=> s.updateSection);
  const first = sections[0];

  return (
    <div className="grid md:grid-cols-2 gap-6 p-6">
      <div>
        <h2 className="text-xl font-semibold mb-3">Outline (V2)</h2>
        <OutlineEditorV2 />
      </div>
      <div>
        <h2 className="text-xl font-semibold mb-3">Structured Templates (V2)</h2>
        {first && <StructuredFormV2 targetSectionId={first.id} />}
        {first && (
          <div className="mt-6">
            <h3 className="font-medium mb-2">Live Preview (first section content)</h3>
            <textarea className="w-full h-48 rounded-xl border p-3" value={first.content} onChange={(e)=> update(first.id, { content: e.target.value })} />
          </div>
        )}
      </div>
    </div>
  );
}

